
function create_model(geom, Eval_mech)

    format long
    clear all;
    close all;
   
    tic;
    import com.comsol.model.*
    import com.comsol.model.util.*

    model = ModelUtil.create('Model');
    model.modelNode.create('comp1');
    geometry = model.geom.create('geom1',3);

    disp('Setting Paramters... ');
        
%%   parameters and definitions
    
    % Scale factors for CAD sweep
    geom.scale_all = 1.02;
    geom.scale_small_holes = 1.0702; % vary scale_small_holes

    % Parameters of the snowflake (slightly modified from Snowflake_topological_insulator paper)
    geom.mech.d = 220e-9;
    geom.mech.r = 180e-9*geom.scale_all;
    geom.mech.delta_r = 0; % Shift the Dirac cone in kx direction
    geom.mech.r1 = geom.mech.r + geom.mech.delta_r;
    geom.mech.r2 = (3*geom.mech.r - geom.mech.r1)/2;
    geom.mech.r3 = (3*geom.mech.r - geom.mech.r1)/2;
    geom.mech.w = 73e-9*geom.scale_all;
    geom.mech.a = 500e-9*geom.scale_all;
    geom.mech.rad_curv = 20e-9*geom.scale_all;
    geom.mech.rad_curv2 = 12e-9*geom.scale_all;
    geom.mech.resolution = 40; % Resolution of the snowflake polygon

    % Minimum distance between optical cavity hole and the snowflake hole
    geom.bridge_width = 70e-9*geom.scale_all;

    % Parameters of the optical cavity
    geom.opt.scale = 1.077*geom.scale_all; % Do not change this scaling parameter for CAD sweeps; Change scale_structure and scale_radius
    geom.opt.delta_a = -9e-9*geom.opt.scale;
    geom.opt.a1 = 425e-9*geom.opt.scale;
    geom.opt.a2 = geom.opt.a1+geom.opt.delta_a;
    geom.opt.a3 = geom.opt.a1+2*geom.opt.delta_a;
    geom.opt.a4 = geom.opt.a1+3*geom.opt.delta_a;
    geom.opt.r = 130e-9*geom.opt.scale*geom.scale_small_holes;
    geom.opt.W = 0;
    geom.opt.h = 130e-9*geom.opt.scale*geom.scale_small_holes;
    geom.opt.b = 150e-9*geom.opt.scale*geom.scale_small_holes;

    geom.opt.nh = 20;
    geom.opt.nv = 40;
    geom.opt.defect_holes = 2*geom.opt.nv;
    geom.opt.defect_width = geom.opt.r/geom.scale_small_holes*2+30e-9;
    geom.opt.defect_length = geom.opt.a1*100;
    geom.opt.cavity_rot = 0;
    geom.opt.cavity_shift = 4; % Cavity shift from the center of triangle
    
    % mass = 0.58
    geom.opt.scale_top_gap = 0.783333; %Scale factor of the radius in one of the triangle

    % Determining the scale factor of the snowflake so that optical cavity fits
    % inside the structure
    n = linspace(1,20,20);
    s = ((n+1)*geom.opt.a1 + 2*geom.opt.a2 + 2*geom.opt.a3 + geom.opt.a4)*sqrt(3)/geom.mech.a;
    index = 14; % 13 bad, 14 good
    num = n(index);
    geom.mech.scale = s(index);

    % Scaling the parameters of the snowflake by the above scale factor
    geom.mech.amech = geom.mech.a*geom.mech.scale;
    geom.mech.r = geom.mech.r*geom.mech.scale;
    geom.mech.r1 = geom.mech.r1*geom.mech.scale;
    geom.mech.r2 = geom.mech.r2*geom.mech.scale;
    geom.mech.r3 = geom.mech.r3*geom.mech.scale;
    geom.mech.w = geom.mech.w*geom.mech.scale;
    geom.mech.a = geom.mech.a*geom.mech.scale;
    geom.mech.rad_curv = geom.mech.rad_curv*geom.mech.scale;
    geom.mech.rad_curv2 = geom.mech.rad_curv2*geom.mech.scale;
    
    % Height of the snowflake triangle (not the triangle where topological edge
    % state is travelling)
    geom.mech.gap_defect_length = geom.mech.amech*sqrt(3)/2 - geom.mech.w*3/2;

    % Mesh parameters
    Eval_mech.hmax = 90e-9;
    Eval_mech.hmin = 25e-9;
    
    % Mirror symmetries
    Eval_mech.symm_z = 1; % about xy plane
    
    % .mph and .mat output files
    Eval_mech.filename = 'experiment_compare_strip_scale_rad_p783_num_domy_6';

    % Strip size
    geom.stack.num_domx = 1; % Number of unit cells in x-direction
    geom.stack.num_domy = 6; % Number of unit cells in y-direction = 2*num_domy+1
    geom.mech.fab_extra = 5.5e-6; % Extra length at the physical edge
    geom.mech.mirror_y = geom.stack.num_domy*geom.mech.amech*sqrt(3)/2; % Position of mirror axis
    
    geom.mech.bdd_shift = 80e-9;
    geom.mech.bdd_width = geom.mech.amech;
    geom.mech.bdd_length = 2*(geom.mech.amech/sqrt(3) + geom.mech.mirror_y + geom.mech.fab_extra);
    
    % Path along the k-space
    Eval_mech.res = 45;
    Eval_mech.kx = linspace(0, pi/(geom.mech.amech)/2, Eval_mech.res);
    Eval_mech.ky = linspace(0, pi/(geom.mech.amech)*sqrt(3)/2, Eval_mech.res);
    
    Eval_mech.NumberOfFrequencies = 25;
    Eval_mech.CentralFrequency = 348e6;
    Eval_mech.bands = zeros(Eval_mech.NumberOfFrequencies,size(Eval_mech.kx,2));
    
%%   COMSOL-GLOBAL PARAMETERS
  
    model.param.set('d', num2str(geom.mech.d));
    model.param.set('r', num2str(geom.mech.r));
    model.param.set('w', num2str(geom.mech.w));
    model.param.set('a', num2str(geom.mech.a));
    model.param.set('a1', num2str(geom.opt.a1));
    model.param.set('a2', num2str(geom.opt.a2));
    model.param.set('a3', num2str(geom.opt.a3));
    model.param.set('a4', num2str(geom.opt.a4));
    model.param.set('r', num2str(geom.opt.r));
    model.param.set('W', num2str(geom.opt.W));
    model.param.set('height', num2str(geom.opt.h));
    model.param.set('b', num2str(geom.opt.b));
    model.param.set('scale_rad', num2str(geom.opt.scale_top_gap));
    model.param.set('hmax', num2str(Eval_mech.hmax));
    model.param.set('hmin', num2str(Eval_mech.hmin));

    mphsave(model, Eval_mech.filename);

%%  COMSOL-GEOMETRY
    
    disp('Building Geometry... ');
    
    workplane = geometry.feature.create('wp1','WorkPlane');
    
    model = draw_snowflakes_hexagon(geom, model, workplane);
    
    model = draw_cavity_hexagon(geom, model, workplane);
    
    model = draw_cavity_hexagon_domain2(geom, model, workplane);
    
    % Bounding surface of the geometry
    pol_x = geom.mech.amech/2 + geom.mech.bdd_shift + [-geom.mech.bdd_width/2 -geom.mech.bdd_width/2 geom.mech.bdd_width/2 geom.mech.bdd_width/2 -geom.mech.bdd_width/2];
    pol_y = geom.mech.mirror_y + [-geom.mech.bdd_length/2 geom.mech.bdd_length/2 geom.mech.bdd_length/2 -geom.mech.bdd_length/2 -geom.mech.bdd_length/2];
    
    [pol_x_rot, pol_y_rot] = rotate(pol_x, pol_y, geom.mech.amech/2 + ...
        geom.mech.bdd_shift, 0, 60);
    
    mech = workplane.geom.create('bound_pol', 'Polygon');
    mech.set('source', 'table');
    for pol_i=1:length(pol_x(1:end-1))
        mech.setIndex('table', num2str(pol_x_rot(pol_i),'%10.9e'), pol_i-1, 0);  %value,row, coloumn
        mech.setIndex('table', num2str(pol_y_rot(pol_i),'%10.9e'), pol_i-1, 1);
    end
    
    workplane.geom.create('uni7', 'Union');
    workplane.geom.feature('uni7').selection('input').set({'uni_snowflakes' 'uni_domain1' 'uni_domain2'});
    
%     workplane.geom.create('uni7', 'Union');
%     workplane.geom.feature('uni7').selection('input').set({'uni_snowflakes'});
    
    workplane.geom.create('dif1', 'Difference');
    workplane.geom.feature('dif1').selection('input').set({'bound_pol'});
    workplane.geom.feature('dif1').selection('input2').set({'uni7'}); 
    
    ext = geometry.feature.create('ext1', 'Extrude');
    ext.set('workplane', 'wp1');
    ext.selection('input').set({'wp1'});
    ext.setIndex('distance', 'd/2', 0);
    
    geometry.run;
    figure(1)
    mphgeom(model);
    view(0,90);
    mphsave(model, Eval_mech.filename)
    
%%   COMSOL-MATERIAL
    
    disp('Setting Material...');
    
    Silicon = model.material.create('mat2', 'Common', 'comp1');

    model.material('mat2').label('Si - Silicon (single-crystal, anisotropic) 1');
    model.material('mat2').set('family', 'custom');
    model.material('mat2').set('lighting', 'cooktorrance');
    model.material('mat2').set('specular', 'custom');
    model.material('mat2').set('customspecular', [0.7843137254901961 1 1]);
    model.material('mat2').set('fresnel', 0.9);
    model.material('mat2').set('roughness', 0.1);
    model.material('mat2').set('shininess', 200);
    model.material('mat2').set('diffuse', 'custom');
    model.material('mat2').set('customdiffuse', [0.6666666666666666 0.6666666666666666 0.7058823529411765]);
    model.material('mat2').set('ambient', 'custom');
    model.material('mat2').set('customambient', [0.6666666666666666 0.6666666666666666 0.7058823529411765]);
    model.material('mat2').set('fresnel', 0.7);
    model.material('mat2').set('roughness', 0.5);
    model.material('mat2').propertyGroup('def').set('density', '2330[kg/m^3]');
    model.material('mat2').propertyGroup.create('Anisotropic', 'Anisotropic');    
    model.material('mat2').propertyGroup('Anisotropic').set('D', {'166[GPa]' '64[GPa]' '166[GPa]' '64[GPa]' '64[GPa]' '166[GPa]' '0[GPa]' '0[GPa]' '0[GPa]' '80[GPa]'  ...
    '0[GPa]' '0[GPa]' '0[GPa]' '0[GPa]' '80[GPa]' '0[GPa]' '0[GPa]' '0[GPa]' '0[GPa]' '0[GPa]'  ...
    '80[GPa]'});
    
    epsilon = 0.05e-9;
    coordBox = [-1, 1; -1, 1; -epsilon, geom.mech.d/2+epsilon];
    si_domain = mphselectbox(model,'geom1',coordBox,'domain');
    Silicon.selection.set(si_domain);

    mphsave(model, Eval_mech.filename);

%%   COMSOL-MESH
    
    disp('Meshing...');
    
    mesh = model.mesh.create('mesh1', 'geom1');
    
    rot = geometry.feature.create('rot1', 'Rotate');
    rot.selection('input').set({'ext1'});
    rot.set('rot', -60);
    rot.set('pos', {num2str(geom.mech.amech/2 + geom.mech.bdd_shift,'%10.9e') '0' '0'});
    
    coordBox = [-1, 1; -1, 1; -epsilon, epsilon];
    sbz = mphselectbox(model,'geom1',coordBox,'boundary');
    
    coordBox = geom.mech.amech/2 + geom.mech.bdd_shift + ...
        [-geom.mech.bdd_width/2 - epsilon, -geom.mech.bdd_width/2 + epsilon; -1, 1; -1, 1];
    l1 = mphselectbox(model,'geom1',coordBox,'boundary');
    l1_all_edge = mphselectbox(model,'geom1',coordBox,'edge');
    
    coordBox = [-epsilon, epsilon; -1, 1; -epsilon, epsilon];
    l1_edge_bot = mphselectbox(model,'geom1',coordBox,'edge');
    
    coordBox = [-epsilon, epsilon; -1, 1; geom.mech.d/2-epsilon, geom.mech.d/2+epsilon];
    l1_edge_top = mphselectbox(model,'geom1',coordBox,'edge');
    
    l1_edge = setdiff(l1_all_edge, horzcat(l1_edge_bot, l1_edge_top));
    
    coordBox = geom.mech.amech/2 + geom.mech.bdd_shift + ...
        [geom.mech.bdd_width/2 - epsilon, geom.mech.bdd_width/2 + epsilon; -1, 1; -1, 1];
    l2 = mphselectbox(model,'geom1',coordBox,'boundary');
    
    freequad = mesh.create('fq1', 'FreeQuad');
    freequad.selection.set([l1]);
    freequad.create('size1', 'Size');
    freequad.feature('size1').set('custom', true);
    freequad.feature('size1').set('hmaxactive', true);
    freequad.feature('size1').set('hmax', 'hmax');
    freequad.feature('size1').set('hminactive', true);
    freequad.feature('size1').set('hmin', 'hmin');

    copy_face = mesh.create('cpf1', 'CopyFace');
    copy_face.selection('source').set([l1]);
    copy_face.selection('destination').set([l2]);
    
    mesh.create('ftri1', 'FreeTri');
    mesh.feature('ftri1').create('size1', 'Size');
    mesh.feature('ftri1').feature('size1').set('custom', 'on');
    mesh.feature('ftri1').feature('size1').set('hmax', 'hmax');
    mesh.feature('ftri1').feature('size1').set('hmin', 'hmin');
    mesh.feature('ftri1').selection.set([sbz]);
    
    mesh.create('swe1', 'Sweep');
    mesh.feature('swe1').create('dis1', 'Distribution');
    mesh.feature('swe1').feature('dis1').set('numelem', num2str(2)); %Number of mesh layers = 2
    
    %%   COMSOL-PHYSICS
    
    disp('Setting Physics...');
    
    physics = model.physics.create('solid', 'SolidMechanics', 'geom1');
    physics.feature('lemm1').set('SolidModel', 'Anisotropic');
    
    coordBox = geom.mech.mirror_y + [-1, 1; -geom.mech.bdd_length/2 - epsilon, -geom.mech.bdd_length/2 + epsilon; -1, 1];
    fixed1 = mphselectbox(model,'geom1',coordBox,'boundary');
    
    coordBox = geom.mech.mirror_y + [-1, 1; geom.mech.bdd_length/2 - epsilon, geom.mech.bdd_length/2 + epsilon; -1, 1];
    fixed2 = mphselectbox(model,'geom1',coordBox,'boundary');
    
    physics.feature.create('pc1', 'PeriodicCondition', 2);    % What does 2 stand for here?
    physics.feature('pc1').set('PeriodicType', 'Floquet');
    physics.feature('pc1').selection.set([l1 l2]);
    
    if Eval_mech.symm_z == +1
        physics.feature.create('sym1', 'SymmetrySolid', 2);
        physics.feature('sym1').selection.set([sbz]);
    elseif Eval_mech.symm_z == -1
        physics.feature.create('as1', 'Antisymmetry', 2);
        physics.feature('as1').selection.set([sbz]);
    end
    
    model.physics('solid').create('fix1', 'Fixed', 2);
    model.physics('solid').feature('fix1').selection.set([fixed1 fixed2]);
  
    mphsave(model, Eval_mech.filename);
    
    rot = geometry.feature.create('rot2', 'Rotate');
    rot.selection('input').set({'rot1'});
    rot.set('rot', 60);
    rot.set('pos', {num2str(geom.mech.amech/2,'%10.9e') '0' '0'});
       
    mphsave(model, Eval_mech.filename);  
    
    % Running mesh
    mesh.run;
    figure(2)
    mphmesh(model)
    mphsave(model, Eval_mech.filename);
    
    Eval_mech.setting_time = toc;
    save(Eval_mech.filename, 'Eval_mech', 'geom');

end